unit pb_main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FileCtrl, checklst, ExtCtrls ;

const
  VERSION_NUMBER = '1.0' ;

type
  TfrmProjectBackup = class(TForm)
    odFindProjectFile: TOpenDialog;
    Panel1: TPanel;
    lblSourceProject: TLabel;
    lbStockUnits: TListBox;
    odFindUnit: TOpenDialog;
    btnExit: TButton;
    StatusMemo: TMemo;
    Splitter1: TSplitter;
    lblTargetPath: TLabel;
    Label2: TLabel;
    btnSource: TButton;
    btnTarget: TButton;
    lblLastBackup: TLabel;
    btnLogFile: TButton;
    lblLogFile: TLabel;
    odFindLogFile: TOpenDialog;
    btnBackup: TButton;
    Splitter2: TSplitter;
    lbExtraFiles: TListBox;
    Label1: TLabel;
    odExtraFiles: TOpenDialog;
    btnClear: TButton;
    lblURL: TLabel;
    Image1: TImage;
    procedure btnBackupClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure lbStockUnitsKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure lbStockUnitsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure btnExitClick(Sender: TObject);
    procedure btnSourceClick(Sender: TObject);
    procedure btnTargetClick(Sender: TObject);
    procedure btnLogFileClick(Sender: TObject);
    procedure lbExtraFilesKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure btnClearClick(Sender: TObject);
    procedure lblURLDblClick(Sender: TObject);
  private
    FInitialTargetDrive : char ;
    FInputPath : string ;
    FTargetPath : string ;
    FLastBackup : string ;
    FLogFile : string ;
    FGlobalIniFile : string ;
    FProjectIniFile : string ;
    FSourceProject : string ;
    FExtraFilesChanged : boolean ;
    FStockUnitsChanged : boolean ;
    FDelphiVersion : integer ;
    slStockUnits : TStringList ;
    slMainProjectUnits : TStringList ;
    slMissingUnits : TStringList ;
    slSearchPaths : TStringList ;
    procedure BuildSearchPathList ;
    procedure DisableControls ;
    procedure EnableControls ;
    procedure EnableBackupButton ;
    procedure ProcessDependencies ;
    procedure ReadSettings( IniFileName : string ; Global : boolean ) ;
    procedure SaveSettings ;
    procedure SetLastBackup( s : string ) ;
    procedure SetLogFile( s : string ) ;
    procedure SetTargetPath( s : string ) ;
    procedure SetProjectIniFile( s : string ) ;
    procedure SetSourceProject( s : string ) ;
  protected
    property LastBackup     : string read FLastBackup write SetLastBackup ;
    property LogFile        : string read FLogFile write SetLogFile ;
    property TargetPath     : string read FTargetPath write SetTargetPath ;
    property ProjectIniFile : string read FProjectIniFile write SetProjectIniFile ;
    property SourceProject  : string read FSourceProject write SetSourceProject ;
  end;

procedure DisplayProjectBackupDialog( ProjectFileName : string ) ;

implementation

uses ShellAPI, IniFiles, Registry ;

const
  INITIAL_LABEL_TEXT   = '<click to select desired folder>' ;
  INITIAL_LABEL_TEXT_2 = '<click to select desired project>' ;
  INITIAL_LABEL_TEXT_3 = '<click to select log file name or folder>';
  SETTINGS_FILENAME = 'ProjectBackup.ini' ;

  {$IFDEF VER130}
  NUM_UNITS = 117 ;
  {$ELSE}
  NUM_UNITS = 99 ;
  {$ENDIF}

  aStockUnits : array[0..NUM_UNITS] of string = (
                'WINDOWS',  'MESSAGES', 'SYSUTILS', 'CLASSES',
                'GRAPHICS', 'CONTROLS', 'FORMS',    'DIALOGS',
                'STDCTRLS', 'COMMDLG',  'BUTTONS',  'PRINTERS',
                'GAUGES',   'LZEXPAND', 'MENUS',    'DATABKR',
                'MXDCUBE',  'MXPBAR',   'MXDSQL',   'IBCONST',
                'MIDASCON', 'DBCGRIDS', 'GRIDS',    'DBGRIDS',
                'ISP3',     'TEEPROCS', 'TEECONST', 'TEENGINE',
                'TEEFUNCI', 'CHART',    'SERIES',   'TEESHAPE',
                'MATH',     'ISAPI',    'ISAPI2',   'NSAPI',
                'HTTPAPP',  'SYNCOBJS', 'COMOBJ',   'DBCLIENT',
                'BDEPROV',  'CGIAPP',   'WEBCONST', 'CHECKLST',
                'EXTDLGS',  'MMSYSTEM', 'JPEG',     'EXPTINTF',
                'TOOLINTF', 'EDITINTF', 'FILEINTF', 'INIFILES',
                'REGISTRY', 'MAPI',     'TLHELP32', 'WININET',
                'COMCTRLS', 'WINSPOOL', 'OLECTRLS', 'WINSOCK',
                'BDE',      'OLECONST', 'SHAREMEM', 'COMSERV',
                'BDECONST', 'QUICKRPT', 'CALENDAR', 'TABNOTBK',
                'SPIN',     'DIROUTLN', 'DBLOOKUP', 'DBTABLES',
                'EXTCTRLS', 'COMMCTRL', 'PICEDIT',  'SHELLAPI',
                'MPLAYER',  'TABS',     'CLIPBRD',  'DDEMAN',
                'OUTLINE',  'DBCONSTS', 'FILECTRL', 'DB',
                'DBCTRLS',  'MASK',     'SHLOBJ',   'COLORGRD',
                'DIROUTLN', 'IBCTRLS',  'IBEVNTS',  'IBPROC32',
                'WINPROCS', 'WINTYPES', 'IMGLIST',  'RICHEDIT',
                'ACTNLIST', 'STDACTNS', 'DBACTNS',  'TOOLWIN'
                {$IFDEF VER130} ,
                'XMLBROKR', 'WEBCOMP',  'ADODB',    'WEBBROKER',
                'ADOINT',   'ADOCONST', 'APPEVNTS', 'COMCORBA',
                'ACCESS97', 'BINDER97', 'DAO97',    'EXCEL97',
                'MSFORMS97','MSPPT8',   'OFFICE97', 'OUTLOOK8',
                'VBIDE97',  'WORD97'
                {$ENDIF}
                ) ;

procedure AddTrailingBackslash(var sPath : string) ; forward ;

{$R *.DFM}

procedure DisplayProjectBackupDialog( ProjectFileName : string ) ;
var
  f : TfrmProjectBackup ;
begin
     f := TfrmProjectBackup.Create(nil) ;
     with f do
        try
           SourceProject := ProjectFileName ; 
           ShowModal ;
        finally
           Release ;
        end ;
end ;

procedure TfrmProjectBackup.btnBackupClick(Sender: TObject);
const
   aProjectExtensions : array[0..4] of string = ( 'dpr', 'dof', 'opt', 'cfg', 'res' ) ;
var
   tf : TextFile ;
   bKeepGoing : boolean ;             
   sData : string ;
   sFileName : string absolute sData ;
   x : integer ;
begin
     // clear histories
     slSearchPaths.Clear ;
     slMainProjectUnits.Clear ;
     slMissingUnits.Clear ;

     // create output directory if necessary
     {$I-}
     MkDir( TargetPath ) ;
     {$I+}
     IOResult ;  // clear flag for next time

     // grab list of stock units out of the listbox
     slStockUnits.Assign( lbStockUnits.Items ) ;

     if ( (not FileExists( TargetPath + ExtractFileName( SourceProject ) ) ) or
        ( MessageDlg( ExtractFileName( SourceProject ) +
                      ' already exists in the ' + TargetPath + ' directory!' + #13 +
                      'Are you sure that you wish to overwrite the previous backup?',
                      mtWarning, [mbOK, mbCancel], 0 ) = mrOK ) ) then begin

        DisableControls ;

        StatusMemo.Lines.Clear ;
        StatusMemo.Update ;

        // header
        StatusMemo.Lines.Add( 'BACKUP STARTED AT ' + DateTimeToStr( Now ) ) ;
        StatusMemo.Lines.Add( SourceProject ) ;
        StatusMemo.Lines.Add( 'Delphi Version #' + IntToStr(FDelphiVersion) ) ;
        StatusMemo.Lines.Add( '' ) ;

        StatusMemo.Lines.Add( 'PROJECT FILES' ) ;

        // backup all project-related files (DPR, OPT, etc)
        for x := 0 to High( aProjectExtensions ) do begin
           sFileName := ChangeFileExt( SourceProject, '.' + aProjectExtensions[ x ] ) ;
           if FileExists( sFileName ) then begin
              StatusMemo.Lines.Add( sFileName ) ;
              CopyFile( PChar( sFileName ), PChar( TargetPath + ExtractFileName( sFileName ) ), FALSE ) ;
           end ;
        end ;

        BuildSearchPathList ;

        StatusMemo.ScrollBars := ssBoth ;

        StatusMemo.Lines.Add( '' ) ;
        StatusMemo.Lines.Add( 'MAIN UNITS' ) ;

        AssignFile( tf, SourceProject ) ;
        {$I-}
        Reset( tf ) ;
        {$I+}
        if IOResult = 0 then begin

           bKeepGoing := True ;
           while bKeepGoing do begin
              ReadLn( tf, sData ) ;
              bKeepGoing := ( Pos( 'USES', UpperCase(sData) ) <> 1 ) ;
           end ;

           // skip to next line
           ReadLn( tf, sData ) ;

           bKeepGoing := True ;
           while bKeepGoing do begin
              ReadLn( tf, sData ) ;
              bKeepGoing := ( Pos( ';', UpperCase(sData) ) = 0 ) ;
              sFileName := Copy( sData, Pos( #39, sData ) + 1, Length( sData ) ) ;
              sFileName := Copy( sFileName, 1, Pos( #39, sFileName ) - 1 ) ;
              slMainProjectUnits.Add( UpperCase( ChangeFileExt( ExtractFileName( sFileName ), '' ) ) ) ;
              StatusMemo.Lines.Add( FInputPath + sFileName) ;
              CopyFile( PChar( FInputPath + sFileName ), PChar( TargetPath + ExtractFileName( sFileName ) ), FALSE ) ;

              // look for and copy accompanying .DFM file
              sFileName := ChangeFileExt( sFileName, '.dfm' ) ;
              if FileExists( FInputPath + sFileName ) and CopyFile( PChar( FInputPath + sFileName ), PChar( TargetPath + ExtractFileName( sFileName ) ), FALSE ) then
                 StatusMemo.Lines.Add( FInputPath + sFileName) ;
           end ;

           CloseFile( tf ) ;

        end ;

        ProcessDependencies ;

        if lbExtraFiles.Items.Count > 0 then begin
           StatusMemo.Lines.Add( '' ) ;
           StatusMemo.Lines.Add( 'EXTRA FILES' ) ;
           for x := 0 to lbExtraFiles.Items.Count - 1 do begin
              sFileName := lbExtraFiles.Items[ x ] ;
              StatusMemo.Lines.Add( sFileName ) ;
              // currently not trapping errors
              CopyFile( PChar( sFileName ), PChar( TargetPath + ExtractFileName( sFileName ) ), FALSE ) ;
           end ;
        end ;

        StatusMemo.Lines.Add( '' ) ;
        sData := 'BACKUP COMPLETED ' ;
        if slMissingUnits.Count > 0 then
           sData := sData + 'WITH FILES MISSING'
        else
           sData := sData + 'SUCCESSFULLY' ;

        sData := sData + ' AT ' + DateTimeToStr( Now ) ;

        StatusMemo.Lines.Add( sData ) ;

        if slMissingUnits.Count > 0 then begin
           StatusMemo.Lines.Add( '' ) ;
           StatusMemo.Lines.Add( 'MISSING FILES:' ) ;
           StatusMemo.Lines.AddStrings( slMissingUnits ) ;
        end ;

        // add footer
        StatusMemo.Lines.BeginUpdate ;
        StatusMemo.Lines.Add( StringOfChar( '=', 60 ) ) ;
        StatusMemo.Lines.Add('Performed by Greg Lief''s Project Backup Expert (version ' + VERSION_NUMBER + ')') ;
        StatusMemo.Lines.Add('Copyright  1999 Greg Lief --- visit http://www.greglief.com') ;
        StatusMemo.Lines.Add('Try the G.L.A.D. components at http://www.greglief.com/glad/') ;
        StatusMemo.Lines.Add( StringOfChar( '=', 60 ) ) ;

        // dump it
        StatusMemo.Lines.SaveToFile( LogFile ) ;

        // remove footer
        for x := 1 to 3 do
           StatusMemo.Lines.Delete( StatusMemo.Lines.Count - 1 ) ;
        StatusMemo.Lines.EndUpdate ;

        SaveSettings ;

        if slMissingUnits.Count = 0 then begin
           MessageBeep( MB_ICONASTERISK ) ;
           MessageDlg( 'Backup completed successfully!', mtInformation, [mbOK], 0 ) ;
        end
        else begin
           MessageBeep( MB_ICONEXCLAMATION ) ;
           MessageDlg( 'Backup completed but some file(s) were not found' + #13 +
                       'Please refer to ' + LogFile + ' for a list of missing files',
                       mtWarning, [mbOK], 0 )
        end ;

        EnableControls ;

     end ;

end ;


procedure TfrmProjectBackup.DisableControls ;
var
   x : integer ;
begin
     for x := 0 to ComponentCount - 1 do
        if Components[x] is TWinControl then
           (Components[x] as TWinControl).Enabled := False ;
end ;


procedure TfrmProjectBackup.EnableControls ;
var
   x : integer ;
begin
     for x := 0 to ComponentCount - 1 do
        if Components[x] is TControl then
           (Components[x] as TControl).Enabled := True ;
end ;


procedure TfrmProjectBackup.ProcessDependencies ;
var
   tf : TextFile ;
   bKeepGoing : boolean ;
   bDependenciesFound : boolean ;
   bFileNameWritten : boolean ;
   sData : string ;
   sTemp : string ;
   sMainFileName : string ;
   sFileName : string ;
   x : integer ;
   iPos : integer ;
   iPos2 : integer ;
   iCurrentLines : integer ;


              procedure WriteThisFileName ;
              begin
                 if not bFileNameWritten then begin
                    bFileNameWritten := True ;
                    StatusMemo.Lines.Add( '' ) ;
                    StatusMemo.Lines.Add( 'DEPENDENCIES FOR ' + sMainFileName) ;
                 end ;
                 bDependenciesFound := True ;
              end ;


              procedure ProcessOneDependency ;
              var
                 y : integer ;
                 sExt : string ;
              begin
                 // make sure this is neither a stock unit nor one of the main project units...
                 if ( slMainProjectUnits.IndexOf( UpperCase( sFileName ) ) = -1 )
                       and ( slStockUnits.IndexOf( UpperCase( sFileName ) ) = -1 ) then begin

                    // loop through all directories in search path
                    y := 0 ;
                    while ( y < slSearchPaths.Count ) and (not (FileExists( slSearchPaths[ y ] + sFileName + '.pas')) )
                          and (not (FileExists( slSearchPaths[ y ] + sFileName + '.dcu')) ) do
                       Inc( y ) ;
                    if ( y < slSearchPaths.Count ) then begin
                       WriteThisFileName ;
                       if FileExists( slSearchPaths[ y ] + sFileName + '.pas') then begin
                          sExt := '.pas' ;
                          StatusMemo.Lines.Add( sFileName + sExt + ' found in ' + slSearchPaths[ y ] ) ;
                       end
                       else begin
                          sExt := '.dcu' ;
                          StatusMemo.Lines.Add( sFileName + '.pas not found, but ' + sFileName + '.dcu found in ' + slSearchPaths[ y ] ) ;
                       end ;

                       CopyFile( PChar( slSearchPaths[ y ] + sFileName + sExt ), PChar( lblTargetPath.Caption + '\' + sFileName + sExt ), FALSE ) ;

                       // also look for accompanying .DFM file
                       if CopyFile( PChar( slSearchPaths[ y ] + sFileName + '.dfm'), PChar( lblTargetPath.Caption + '\' + sFileName + '.dfm' ), FALSE ) then
                          StatusMemo.Lines.Add( sFileName + '.dfm found in ' + slSearchPaths[ y ] ) ;

                    end
                    else begin
                       if slMissingUnits.IndexOf( '* ' + sFileName ) = -1 then
                          slMissingUnits.Add( '* ' + sFileName ) ;
                       WriteThisFileName ;
                       StatusMemo.Lines.Add( sFileName + ' not found in search path' ) ;
                    end ;
                 end ;
              end ;


              procedure MainLoop ;
              var
                 z : integer ;
                 bComment : boolean ;
              begin
                 bKeepGoing := True ;
                 while bKeepGoing do begin
                    bKeepGoing := ( Pos( ';', sData ) = 0 ) ;
                    iPos := Pos( ',', sData ) ;
                    bComment := False ;
                    while iPos > 0 do begin
                       sFileName := Trim( Copy( sData, 1, iPos - 1 ) ) ;
                       // did we stumble over an in-line comment?
                       iPos2 := Pos( '{', sFileName ) ;
                       bComment := ( iPos2 <> 0 ) ;
                       if bComment and ( iPos2 > 1 ) then
                          sFileName := Copy( sFileName, 1, iPos2 - 1 ) ;

                       if ( sFileName <> '' ) and ( ( not bComment ) or
                                                    ( bComment and ( iPos2 > 1 ) ) ) then
                          ProcessOneDependency ;

                       // if all that is left in this line is a comment, punt!
                       if sData[ iPos ] = ';' then begin
                          sData := '' ;
                          iPos := 0
                       end
                       else begin

                          // if we stumbled over an in-line comment,
                          // look for the closing brace
                          if bComment then begin
                             iPos := Pos( '}', sData ) ;
                             if iPos = 0 then
                                sData := ''
                          end ;

                          if iPos <> 0 then begin

                             sData := Copy( sData, iPos + 1, Length( sData ) ) ;

                             iPos := Pos( ',', sData ) ;

                             // ensure that there is not a semi-colon before the next comma,
                             // which would indicate that there is a comment at the end of this line!
                             z := Pos( ';', sData ) ;
                             if ( z <> 0 ) and ( z < iPos ) then
                                iPos := z ;

                          end ;

                       end ;

                    end ;

                    // be sure to check last unit in list
                    sData := Trim( sData ) ;
                    // ignore blank lines and comments
                    if ( sData <> '' ) and ( Pos( '//', sData ) <> 1 ) and
                                           ( Pos( '*',  sData ) <> 1 ) and
                                           ( Pos( '{',  sData ) <> 1 ) then begin
                       iPos := Pos( ';', sData ) ;
                       if iPos > 0 then
                          sFileName := Trim( Copy( sData, 1, iPos - 1 ) )
                       else
                          sFileName := Trim( sData ) ;
                       if sFileName <> '' then
                          ProcessOneDependency ;
                    end ;

                    ReadLn( tf, sData ) ;

                    // if we are still in an in-line comment,
                    // keep looking for closing brace!
                    if bComment then begin
                       iPos := Pos( '}', sData ) ;
                       while iPos = 0 do begin
                          ReadLn( tf, sData ) ;
                          iPos := Pos( '}', sData ) ;
                       end ;
                       sData := Copy( sData, iPos + 1, Length( sData ) ) ;
                    end ;

                 end ;
              end ;


begin
     // save current # of lines in memo so that we don't
     // get recursive processing!
     iCurrentLines := StatusMemo.Lines.Count - 1 ;
     bDependenciesFound := False ;
     for x := 0 to iCurrentLines do begin
        sMainFileName := StatusMemo.Lines[ x ] ;
        bFileNameWritten := False ;
        if UpperCase( ExtractFileExt( sMainFileName ) ) = '.PAS' then begin

           AssignFile( tf, sMainFileName ) ;
           {$I-}
           Reset( tf ) ;
           {$I+}
           if IOResult = 0 then begin
              // find USES clause in INTERFACE section
              bKeepGoing := True ;
              while bKeepGoing do begin
                 ReadLn( tf, sData ) ;
                 bKeepGoing := ( Pos( 'INTERFACE', UpperCase(sData) ) <> 1 ) ;
              end ;

              bKeepGoing := True ;
              while bKeepGoing do begin
                 ReadLn( tf, sData ) ;
                 bKeepGoing := ( Pos( 'USES', UpperCase(sData) ) <> 1 ) and
                               ( Pos( 'IMPLEMENTATION', UpperCase(sData) ) <> 1 ) ;
              end ;

              // found USES clause in INTERFACE section
              if Pos( 'USES', UpperCase( sData ) ) = 1 then begin
                 // strip off "USES"
                 sData := Copy( sData, 6, Length( sData ) ) ;
                 if sData = '' then
                    ReadLn( tf, sData ) ;
                 MainLoop ;

                 // move to beginning of IMPLEMENTATION section
                 bKeepGoing := True ;
                 while bKeepGoing do begin
                    ReadLn( tf, sData ) ;
                    bKeepGoing := ( Pos( 'IMPLEMENTATION', UpperCase(sData) ) <> 1 ) ;
                 end ;

              end ;

              // find USES clause in IMPLEMENTATION section
              bKeepGoing := True ;
              while bKeepGoing do begin
                 ReadLn( tf, sData ) ;
                 sTemp := UpperCase( sData ) ;
                 bKeepGoing := ( Pos( 'USES', sTemp ) <> 1 ) and
                               ( Pos( 'FUNCTION', sTemp ) <> 1 ) and
                               ( Pos( 'PROCEDURE', sTemp ) <> 1 ) and
                               ( Pos( 'END.', sTemp ) <> 1 ) ;
              end ;

              if Pos( 'USES', sTemp ) = 1 then begin
                 // strip off "USES"
                 sData := Copy( sData, 6, Length( sData ) ) ;
                 MainLoop ;
              end ;

              CloseFile( tf ) ;

           end ;

        end ;

     end ;

     if not bDependenciesFound  then begin
        StatusMemo.Lines.Add( '' ) ;
        StatusMemo.Lines.Add( 'NO DEPENDENCIES FOUND' )
     end ;

end ;

procedure TfrmProjectBackup.FormCreate(Sender: TObject);
var
   x : integer ;
   iDriveType : integer ;
   bKeepGoing : boolean ;
begin
     // set initial Delphi version
     {$IFDEF VER120}
     FDelphiVersion := 4   ;
     {$ELSE}
       {$IFDEF VER130}
       FDelphiVersion := 5 ;
       {$ELSE}
       FDelphiVersion := 3 ;
       {$ENDIF}
     {$ENDIF}

     // find first floppy or network drive
     FInitialTargetDrive := 'A' ;
     bKeepGoing := True ;
     while ( FInitialTargetDrive <= 'Z' ) and bKeepGoing do begin
        iDriveType := GetDriveType( PChar( FInitialTargetDrive + ':\' ) ) ;
        bKeepGoing := ( iDriveType <> DRIVE_REMOVABLE ) and
                      ( iDriveType <> DRIVE_REMOTE ) ;
     end ;

     // initialize "blank" labels
     lblSourceProject.Caption := INITIAL_LABEL_TEXT_2 ;
     lblTargetPath.Caption    := INITIAL_LABEL_TEXT ;
     lblLogFile.Caption       := INITIAL_LABEL_TEXT_3;

     // set filename for global INI
     FGlobalIniFile := ExtractFilePath( ParamStr( 0 ) ) + SETTINGS_FILENAME ;

     FStockUnitsChanged := False ;
     FExtraFilesChanged := False ;

     slStockUnits := TStringList.Create ;
     for x := 0 to High( aStockUnits ) do
        slStockUnits.Add( aStockUnits[x] ) ;
     slStockUnits.Sorted := True ;

     slSearchPaths := TStringList.Create ;
     slMainProjectUnits := TStringList.Create ;
     slMissingUnits := TStringList.Create ;

     // copy stock units to listbox for user modification
     lbStockUnits.Items.Assign( slStockUnits ) ;

     ReadSettings( FGlobalIniFile, TRUE ) ;

     Application.HintHidePause := 5000 ;

     lbStockUnits.Hint := 'Units listed here will be excluded from backups performed' + #13 +
                          'during this session. Click this listbox, then press INSERT' + #13 +
                          'or DELETE to exclude/include other units as necessary.' ;

     lbExtraFiles.Hint := 'Any files listed here will be included in backups performed' + #13 +
                          'during this session. Click this listbox, then press INSERT' + #13 +
                          'or DELETE to add or remove files as necessary.' ;

end;

procedure TfrmProjectBackup.FormClose(Sender: TObject; var Action: TCloseAction);
begin
     slMainProjectUnits.Free ;
     slStockUnits.Free ;
     slSearchPaths.Free ;
     slMissingUnits.Free ;
end;


procedure TfrmProjectBackup.BuildSearchPathList ;
const
   D4_ROOT_SYMBOL = '$(DELPHI)' ;
var
   riTemp : TRegIniFile ;
   sPathData : string ;
   sRootPath : string ;
   iPos : integer ;
   ifTemp : TIniFile ;

                        procedure ParsePath ;
                        var
                           sTemp : string ;
                        begin
                            // parse each directory in search path
                            iPos := Pos( ';', sPathData ) ;
                            while iPos > 0 do begin
                               sTemp := Copy( sPathData, 1, iPos - 1 ) ;
                               // replace global D4 root directory if present
                               if Pos( D4_ROOT_SYMBOL, sTemp ) = 1 then begin
                                  {$IFDEF VER100}  // D3 doesn't have the StringReplace function
                                  sTemp := sRootPath + Copy( sTemp, Length(D4_ROOT_SYMBOL) + 1, Length( sTemp ) ) ;
                                  {$ELSE}
                                  sTemp := StringReplace( sTemp, D4_ROOT_SYMBOL, sRootPath, [rfIgnoreCase] ) ;
                                  {$ENDIF}
                               end ;
                               AddTrailingBackslash( sTemp ) ;
                               slSearchPaths.Add( sTemp ) ;
                               sPathData := Copy( sPathData, iPos + 1, Length( sPathData ) ) ;
                               iPos := Pos( ';', sPathData ) ;
                            end ;
                            // don't forget the last one
                            if sPathData <> '' then begin
                               AddTrailingBackslash( sPathData ) ;
                               slSearchPaths.Add( sPathData ) ;
                            end ;
                        end ;


begin
     // first add source directory
     slSearchPaths.Add( ExtractFilePath( SourceProject ) ) ;

     // look for .DOF file, which might contain project-specific search path
     if FileExists( ChangeFileExt( SourceProject, '.DOF' ) ) then begin
        ifTemp := TIniFile.Create( ChangeFileExt( SourceProject, '.DOF' ) ) ;
        sPathData := ifTemp.ReadString( 'Directories', 'SearchPath', '' ) ;
        ifTemp.Free ;
        ParsePath ;
     end ;

     // now pull default search path from registry
     riTemp := TRegIniFile.Create( 'Software' ) ;
     riTemp.OpenKey( 'Borland\Delphi\' + IntToStr( FDelphiVersion ) + '.0', FALSE ) ;
     if FDelphiVersion > 3 then begin    // D4 and D5 specific logic
        sPathData := riTemp.ReadString( 'Library', 'Search Path', '' ) ;  // "Search Path" rather than "SearchPath"
        // retrieve global D4/D5 root path so that we can expand the $(DELPHI) symbol
        riTemp.CloseKey ;
        riTemp.OpenKey( 'Software\Borland\Package Collection Editor\Installed Programs', FALSE ) ;
        sRootPath := ( riTemp.ReadString( 'Delphi', 'Path', '' ) ) ;
     end
     else
        sPathData := riTemp.ReadString( 'Library', 'SearchPath', '' ) ;  // D2/D3
     riTemp.CloseKey ;
     riTemp.Free ;

     ParsePath ;

end ;


procedure TfrmProjectBackup.lbStockUnitsKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
   x : integer ;
begin
     case Key of

        VK_INSERT : if odFindUnit.Execute then begin
                       FStockUnitsChanged := True ;
                       for x := 0 to odFindUnit.Files.Count - 1 do
                          lbStockUnits.Items.Add( UpperCase ( ChangeFileExt( ExtractFileName( odFindUnit.Files[ x ] ), '' ) ) ) ;
                    end ;

        VK_DELETE : if MessageDlg( 'Remove ' + lbStockUnits.Items[ lbStockUnits.ItemIndex ] +
                                   ' from list of excluded units?', mtConfirmation,
                                   [mbOK, mbCancel], 0 ) = mrOK then begin
                       FStockUnitsChanged := True ;
                       lbStockUnits.Items.Delete( lbStockUnits.ItemIndex ) ;
                    end ;

     end ;
end;

procedure TfrmProjectBackup.lbStockUnitsDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
begin
     with Control as TListBox do begin
        Canvas.FillRect( Rect ) ;

        // draw user-defined stock units in a different color
        if slStockUnits.IndexOf( Items[Index] ) = -1 then
           if odSelected in State then
              Canvas.Font.Color := clAqua
           else
              Canvas.Font.Color := clRed ;

        Canvas.TextOut( Rect.Left + 1, Rect.Top + 1, Items[Index] ) ;
     end ;
end;

procedure TfrmProjectBackup.btnExitClick(Sender: TObject);
begin
     Close ;
end;

procedure AddTrailingBackslash(var sPath : string) ;
begin
     if sPath[ Length( sPath) ] <> '\' then
        sPath := sPath + '\' ;
end ;

procedure TfrmProjectBackup.btnSourceClick(Sender: TObject);
begin
     if odFindProjectFile.Execute then begin
        SourceProject := odFindProjectFile.FileName ;
        EnableBackupButton ;
     end ;
end;

procedure TfrmProjectBackup.btnTargetClick(Sender: TObject);
var
   s : string ;
const
   DIR_NAME = 'GregTest' ;
begin
     if TargetPath = '' then begin
        s := FInitialTargetDrive + ':\' ;
        {$I-}
        MkDir( s + DIR_NAME ) ;
        RmDir( s + DIR_NAME ) ;
        {$I+}
        if IOResult <> 0 then
           s := ExtractFileDrive( ParamStr( 0 ) ) + '\' ;
     end
     else
        s := TargetPath ;

     if SelectDirectory( s, [sdAllowCreate, sdPrompt], 0 ) then
        if s = FInputPath then
           MessageDlg('Target directory must be different than source directory!', mtError, [mbOK], 0)
        else begin
           AddTrailingBackslash( s ) ;
           TargetPath := s ;
           lblTargetPath.Caption := TargetPath ;
           EnableBackupButton ;
        end ;

end;

procedure TfrmProjectBackup.EnableBackupButton ;
begin
     btnBackup.Enabled := ( SourceProject <> '' ) and ( TargetPath <> INITIAL_LABEL_TEXT ) ;
end ;

procedure TfrmProjectBackup.ReadSettings( IniFileName : string ; Global : boolean ) ;
var
   t : TIniFile ;
   s : string ;
   x : integer ;
begin
     t := TIniFile.Create( IniFileName ) ;
     SourceProject := t.ReadString( 'Settings', 'SourceProject', '' ) ;

     if not Global then begin

        TargetPath := t.ReadString( 'Settings', 'TargetPath', INITIAL_LABEL_TEXT ) ;

        // read in list of excluded units if present in .INI file
        s := t.ReadString( 'Excluded Units', 'Unit0', '' ) ;
        if s <> '' then begin
           x := 0 ;
           lbStockUnits.Items.Clear ;
           while s <> '' do begin
              lbStockUnits.Items.Add( s ) ;
              Inc( x ) ;
              s := t.ReadString( 'Excluded Units', 'Unit' + IntToStr( x ), '' ) ;
           end ;
        end ;

        // read in list of extra files units if present in .INI file
        s := t.ReadString( 'Extra Files', 'File0', '' ) ;
        if s <> '' then begin
           x := 0 ;
           lbExtraFiles.Items.Clear ;
           while s <> '' do begin
              lbExtraFiles.Items.Add( s ) ;
              Inc( x ) ;
              s := t.ReadString( 'Extra Files', 'File' + IntToStr( x ), '' ) ;
           end ;
        end ;

        s := t.ReadString('Settings', 'LastBackup', '' ) ;
        if s <> '' then
           LastBackup := s ;

        t.Free ;

     end ;

     EnableBackupButton ;
end ;

procedure TfrmProjectBackup.SaveSettings ;
var
   t : TIniFile ;
   x : integer ;
begin
     // first write global INI file
     t := TIniFile.Create( FGlobalIniFile ) ;
     t.WriteString( 'Settings', 'SourceProject', SourceProject ) ;
     t.Free ;

     // then write project-specific INI file
     t := TIniFile.Create( ProjectIniFile ) ;
     t.WriteString( 'Settings', 'SourceProject', SourceProject ) ;
     t.WriteString( 'Settings', 'TargetPath', TargetPath ) ;
     LastBackup := DateTimeToStr( Now ) ;
     t.WriteString('Settings', 'LastBackup', LastBackup ) ;

     if FStockUnitsChanged then begin
        for x := 0 to lbStockUnits.Items.Count - 1 do
           t.WriteString( 'Excluded Units', 'Unit' + IntToStr(x), lbStockUnits.Items[ x ] ) ;
     end ;

     if FExtraFilesChanged then begin
        for x := 0 to lbExtraFiles.Items.Count - 1 do
           t.WriteString( 'Extra Files', 'File' + IntToStr(x), lbExtraFiles.Items[ x ] ) ;
     end ;

     t.Free ;
end ;

procedure TfrmProjectBackup.SetLastBackup( s : string ) ;
begin
     FLastBackup := s ;
     lblLastBackup.Caption := 'Last backup: ' + FLastBackup ;
end ;

procedure TfrmProjectBackup.SetTargetPath( s : string ) ;
begin
     if s <> FTargetPath then begin
        FTargetPath := s ;
        lblTargetPath.Caption := s ;
     end ;
end ;

procedure TfrmProjectBackup.SetSourceProject( s : string ) ;
begin
     if s <> FSourceProject then begin
        FSourceProject := s ;
        if s = '' then
           lblSourceProject.Caption := INITIAL_LABEL_TEXT_2
        else
           lblSourceProject.Caption := s ;
        FInputPath := UpperCase( ExtractFilePath( SourceProject ) ) ;
        ProjectIniFile := UpperCase( ChangeFileExt( SourceProject, '.ini' ) ) ;
        if LogFile = '' then
           LogFile := ChangeFileExt( SourceProject, '_backup.log' ) ;
     end ;
end ;

procedure TfrmProjectBackup.SetLogFile( s : string ) ;
begin
     if s <> FLogFile then begin
        FLogFile := s ;
        lblLogFile.Caption := s ;
     end ;
end ;

procedure TfrmProjectBackup.SetProjectIniFile( s : string ) ;
begin
     FProjectIniFile := s ;
     if FileExists( s ) then
        ReadSettings( s, False )
     // no project-specific INI file... clear previous settings accordingly
     else begin
        LastBackup := '<none>' ;
        TargetPath := INITIAL_LABEL_TEXT ;
        LogFile := '' ;
        lbStockUnits.Items.Assign( slStockUnits ) ;
     end ;
end ;

procedure TfrmProjectBackup.btnLogFileClick(Sender: TObject);
begin
     odFindLogFile.FileName := ExtractFileName(lblLogFile.Caption);
     if odFindLogFile.Execute then
        LogFile := odFindLogFile.FileName ;
end;

procedure TfrmProjectBackup.lbExtraFilesKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
   iWidth : integer ;
   x : integer ;
const
   PADDING = 5 ;
begin
     case Key of
        VK_INSERT : if odExtraFiles.Execute then begin
                       FExtraFilesChanged := True ;
                       for x := 0 to odExtraFiles.Files.Count - 1 do
                          lbExtraFiles.Items.Add( odExtraFiles.Files[ x ] ) ;
                       // if newly added filename is too wide to fit in listbox,
                       // adjust horizontal scrolling accordingly
                       iWidth := lbExtraFiles.Canvas.TextWidth( odExtraFiles.FileName ) ;
                       if iWidth + PADDING > lbExtraFiles.Perform( LB_GETHORIZONTALEXTENT, 0, 0 ) then
                          lbExtraFiles.Perform( LB_SETHORIZONTALEXTENT, iWidth + PADDING, 0 ) ;
                       odExtraFiles.FileName := '' ;
                       odExtraFiles.InitialDir := FInputPath ;
                    end ;

        VK_DELETE : if MessageDlg( 'Remove ' + lbExtraFiles.Items[ lbExtraFiles.ItemIndex ] +
                                   ' from list of extra files?', mtConfirmation,
                                   [mbOK, mbCancel], 0 ) = mrOK then begin
                       FExtraFilesChanged := True ;
                       // redo horizontal scrolling if we deleted the longest element
                       (*
                       iWidth := lbExtraFiles.Items[ lbExtraFiles.ItemIndex ] ;
                       if iWidth + PADDING = lbExtraFiles.Perform( LB_GETHORIZONTALEXTENT, 0, 0 ) then
                          lbExtraFiles.Perform( LB_SETHORIZONTALEXTENT, iWidth + PADDING, 0 ) then
                       *)
                       lbExtraFiles.Items.Delete( lbExtraFiles.ItemIndex ) ;
                    end ;

     end ;
end;

procedure TfrmProjectBackup.btnClearClick(Sender: TObject);
begin
     if MessageDlg( 'Clear all backup settings?', mtConfirmation, [mbOK, mbCancel], 0 ) = mrOK then begin
        SourceProject  := '' ;
        LogFile        := '' ;
        TargetPath     := '' ;
        ProjectIniFile := '' ;
        lbExtraFiles.Items.Clear ;
        FStockUnitsChanged := False ;
        FExtraFilesChanged := False ;
        lbStockUnits.Items.Assign( slStockUnits ) ;

        lblSourceProject.Caption := INITIAL_LABEL_TEXT_2;
        lblTargetPath.Caption := INITIAL_LABEL_TEXT;
        lblLogFile.Caption := INITIAL_LABEL_TEXT_3;

     end ;
end;

procedure TfrmProjectBackup.lblURLDblClick(Sender: TObject);
begin
     ShellExecute(0, 'open', 'http://www.greglief.com/delphi', nil, nil, SW_SHOWNORMAL) ;
end;

end.
